<?php
/**
 * Plugin Name: Auto Generate Content 
 * Description: Template konten pakai {title} + spintax {a|b|c}
 * Version: 1.1.0
 * Author: EMSIWA
 */

if (!defined('ABSPATH')) exit;

class ACG_Plugin {
  const OPT_TEMPLATE = 'acg_template_content';

  const CPT = 'acg_campaign';

  // Meta keys campaign
  const MK_KEYWORDS    = '_acg_keywords';
  const MK_POST_TYPE   = '_acg_post_type';
  const MK_AMOUNT      = '_acg_amount';
  const MK_CATEGORY    = '_acg_category';
  const MK_POST_STATUS = '_acg_post_status';

  // Schedule
  const MK_SCHEDULE_VALUE = '_acg_schedule_value';
  const MK_SCHEDULE_UNIT  = '_acg_schedule_unit';

  // Tracking
  const MK_USED_TITLES = '_acg_used_titles';
  const MK_GEN_IDS     = '_acg_generated_post_ids';

  public function __construct() {
    add_action('init', [$this, 'register_cpt']);
    add_action('admin_menu', [$this, 'register_menus']);
    add_action('admin_init', [$this, 'register_settings']);

    add_action('add_meta_boxes', [$this, 'add_campaign_metaboxes']);
    add_action('save_post_' . self::CPT, [$this, 'save_campaign_meta'], 10, 2);

    // Generate otomatis saat campaign dipublish / diupdate dalam status publish
    add_action('save_post_' . self::CPT, [$this, 'maybe_generate_on_save'], 20, 2);

    // Cron tick handler
    add_action('acg_run_campaign_tick', [$this, 'cron_run_campaign_tick'], 10, 1);

    // Cleanup cron ketika campaign dihapus
    add_action('before_delete_post', [$this, 'cleanup_campaign_cron_on_delete']);
  }

  public function register_cpt() {
    $labels = [
      'name'               => 'Campaign',
      'singular_name'      => 'Campaign',
      'add_new'            => 'Add New',
      'add_new_item'       => 'Add New Campaign',
      'edit_item'          => 'Edit Campaign',
      'new_item'           => 'New Campaign',
      'view_item'          => 'View Campaign',
      'search_items'       => 'Search Campaign',
      'not_found'          => 'No campaigns found',
      'not_found_in_trash' => 'No campaigns found in Trash',
      'menu_name'          => 'Auto Generate Content',
    ];

    register_post_type(self::CPT, [
      'labels'             => $labels,
      'public'             => false,
      'show_ui'            => true,
      'show_in_menu'       => 'acg_main_menu',
      'menu_position'      => 58,
      'supports'           => ['title'],
      'capability_type'    => 'post',
      'hierarchical'       => false,
      'has_archive'        => false,
      'rewrite'            => false,
    ]);
  }

  public function register_menus() {
    // Menu utama: Auto Generate Content
    add_menu_page(
      'Auto Generate Content',
      'Auto Generate Content',
      'manage_options',
      'acg_main_menu',
      function() {
        wp_safe_redirect(admin_url('edit.php?post_type=' . self::CPT));
        exit;
      },
      'dashicons-megaphone',
      58
    );

    // Submenu: Campaign list
    add_submenu_page(
      'acg_main_menu',
      'Campaign',
      'Campaign',
      'manage_options',
      'edit.php?post_type=' . self::CPT
    );

    // Submenu: Template Konten
    add_submenu_page(
      'acg_main_menu',
      'Template Konten',
      'Template Konten',
      'manage_options',
      'acg_template_menu',
      [$this, 'render_template_page']
    );
  }

  public function register_settings() {
    register_setting('acg_template_group', self::OPT_TEMPLATE, [
      'type' => 'string',
      'sanitize_callback' => [$this, 'sanitize_template'],
      'default' => "<h2>{title}</h2>\n<p>{Halo|Hai|Selamat datang}! Ini artikel tentang {title}.</p>\n<p>{Semoga|Mudah-mudahan} bermanfaat.</p>"
    ]);
  }

  public function sanitize_template($value) {
    $value = str_replace(["\r\n", "\r"], "\n", (string)$value);
    return trim($value);
  }

  public function render_template_page() {
    if (!current_user_can('manage_options')) return;

    $template = get_option(self::OPT_TEMPLATE, '');

    ?>
    <div class="wrap">
      <h1>Template Konten</h1>
      <p>Gunakan placeholder: <code>{title}</code></p>
      <p>Spintax: <code>{opsi1|opsi2|opsi3}</code></p>

      <form method="post" action="options.php">
        <?php settings_fields('acg_template_group'); ?>

        <textarea
          name="<?php echo esc_attr(self::OPT_TEMPLATE); ?>"
          rows="14"
          style="width:100%; font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, 'Liberation Mono', 'Courier New', monospace;"
        ><?php echo esc_textarea($template); ?></textarea>

        <?php submit_button('Simpan Template'); ?>
      </form>

      <h2>Contoh Template</h2>
      <pre style="background:#fff; padding:12px; border:1px solid #ddd; white-space:pre-wrap;"><?php
echo esc_html("<h2>{title}</h2>
<p>{Halo|Hai|Selamat datang}! Ini artikel tentang {title}.</p>
<p>{Di artikel ini|Di pembahasan ini} kita bahas {tips|cara|strategi} seputar {title}.</p>
<ul>
  <li>{Tips|Cara|Trik} #1 untuk {title}</li>
  <li>{Tips|Cara|Trik} #2 untuk {title}</li>
</ul>
<p>{Semoga|Mudah-mudahan} bermanfaat.</p>");
      ?></pre>
    </div>
    <?php
  }

  public function add_campaign_metaboxes() {
    add_meta_box(
      'acg_keywords_box',
      'Kata Kunci Posting Masal',
      [$this, 'render_keywords_metabox'],
      self::CPT,
      'normal',
      'high'
    );

    add_meta_box(
      'acg_settings_box',
      'Pengaturan Kampanye',
      [$this, 'render_settings_metabox'],
      self::CPT,
      'normal',
      'default'
    );
  }

  public function render_keywords_metabox($post) {
    wp_nonce_field('acg_campaign_save', 'acg_campaign_nonce');

    $keywords = get_post_meta($post->ID, self::MK_KEYWORDS, true);
    if (!is_string($keywords)) $keywords = '';

    ?>
    <p><strong>Kata Kunci:</strong></p>
    <textarea name="acg_keywords" rows="10" style="width:100%;"><?php echo esc_textarea($keywords); ?></textarea>
    <p style="margin-top:6px; color:#555;">
      Isi dengan kata kunci / judul. Kata kunci akan diposting otomatis saat kampanye dipublish.
      Silakan tambahkan kata kunci per baris misalnya:<br>
      Kata kunci 1<br>
      Kata kunci 2<br>
      Kata kunci 3
    </p>
    <?php
  }

  public function render_settings_metabox($post) {
    $saved_post_type = get_post_meta($post->ID, self::MK_POST_TYPE, true);
    $saved_amount    = get_post_meta($post->ID, self::MK_AMOUNT, true);
    $saved_cat       = get_post_meta($post->ID, self::MK_CATEGORY, true);
    $saved_status    = get_post_meta($post->ID, self::MK_POST_STATUS, true);

    $schedule_value  = get_post_meta($post->ID, self::MK_SCHEDULE_VALUE, true);
    $schedule_unit   = get_post_meta($post->ID, self::MK_SCHEDULE_UNIT, true);

    if (!$saved_post_type) $saved_post_type = 'post';
    if (!$saved_amount)    $saved_amount = 1;
    if (!$saved_status)    $saved_status = 'publish';

    if (!$schedule_value)  $schedule_value = 24;
    if (!$schedule_unit)   $schedule_unit = 'hours';

    // Post types (public UI)
    $pts = get_post_types(['public' => true], 'objects');

    // Categories
    $cats = get_categories(['hide_empty' => false]);

    ?>
    <table class="form-table" role="presentation">
      <tr>
        <th scope="row"><label for="acg_post_type">Tipe Post</label></th>
        <td>
          <select name="acg_post_type" id="acg_post_type" style="min-width:260px;">
            <?php foreach ($pts as $pt): ?>
              <option value="<?php echo esc_attr($pt->name); ?>" <?php selected($saved_post_type, $pt->name); ?>>
                <?php echo esc_html($pt->labels->singular_name); ?>
              </option>
            <?php endforeach; ?>
          </select>
          <p class="description">Silakan pilih jenis pos di sini (post/halaman/custom post type publik).</p>
        </td>
      </tr>

      <tr>
        <th scope="row"><label for="acg_amount">Jumlah</label></th>
        <td>
          <input type="number" min="1" name="acg_amount" id="acg_amount" value="<?php echo esc_attr((int)$saved_amount); ?>" style="width:120px;">
          <p class="description">Berapa banyak posting yang ingin diposting segera setelah mempublikasikan kampanye ini (post pertama dibuat saat publish, sisanya dijadwalkan).</p>
        </td>
      </tr>

      <tr>
        <th scope="row"><label for="acg_category">Kategori</label></th>
        <td>
          <select name="acg_category" id="acg_category" style="min-width:260px;">
            <option value="">— Pilih Kategori —</option>
            <?php foreach ($cats as $c): ?>
              <option value="<?php echo esc_attr($c->term_id); ?>" <?php selected((string)$saved_cat, (string)$c->term_id); ?>>
                <?php echo esc_html($c->name); ?>
              </option>
            <?php endforeach; ?>
          </select>
          <p class="description">Kategori posting untuk kampanye ini (dipakai jika post type mendukung category).</p>
        </td>
      </tr>

      <tr>
        <th scope="row"><label for="acg_post_status">Status post</label></th>
        <td>
          <select name="acg_post_status" id="acg_post_status" style="min-width:260px;">
            <option value="publish" <?php selected($saved_status, 'publish'); ?>>Publish</option>
            <option value="draft"   <?php selected($saved_status, 'draft'); ?>>Draft</option>
            <option value="pending" <?php selected($saved_status, 'pending'); ?>>Pending Review</option>
          </select>
          <p class="description">Status posting yang akan dibuat oleh kampanye ini.</p>
        </td>
      </tr>

      <tr>
        <th scope="row"><label for="acg_schedule_value">Jadwal Auto Posting</label></th>
        <td>
          <input type="number" min="1" name="acg_schedule_value" id="acg_schedule_value"
                 value="<?php echo esc_attr((int)$schedule_value); ?>" style="width:120px;">

          <label style="margin-left:12px;">
            <input type="radio" name="acg_schedule_unit" value="seconds" <?php checked($schedule_unit, 'seconds'); ?>>
            seconds
          </label>

          <label style="margin-left:10px;">
            <input type="radio" name="acg_schedule_unit" value="hours" <?php checked($schedule_unit, 'hours'); ?>>
            hours
          </label>

          <label style="margin-left:10px;">
            <input type="radio" name="acg_schedule_unit" value="days" <?php checked($schedule_unit, 'days'); ?>>
            days
          </label>

          <p class="description">Silahkan mengisi waktu atau rentang waktu antara setiap auto pos.</p>
        </td>
      </tr>
    </table>
    <?php
  }

  public function save_campaign_meta($post_id, $post) {
    if (!isset($_POST['acg_campaign_nonce']) || !wp_verify_nonce($_POST['acg_campaign_nonce'], 'acg_campaign_save')) {
      return;
    }
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    // Keywords
    $keywords = isset($_POST['acg_keywords']) ? (string)$_POST['acg_keywords'] : '';
    $keywords = str_replace(["\r\n", "\r"], "\n", $keywords);
    $keywords = trim($keywords);
    update_post_meta($post_id, self::MK_KEYWORDS, $keywords);

    // Settings
    $post_type = isset($_POST['acg_post_type']) ? sanitize_key($_POST['acg_post_type']) : 'post';
    update_post_meta($post_id, self::MK_POST_TYPE, $post_type);

    $amount = isset($_POST['acg_amount']) ? max(1, (int)$_POST['acg_amount']) : 1;
    update_post_meta($post_id, self::MK_AMOUNT, $amount);

    $cat = isset($_POST['acg_category']) ? (int)$_POST['acg_category'] : 0;
    update_post_meta($post_id, self::MK_CATEGORY, $cat);

    $status = isset($_POST['acg_post_status']) ? sanitize_key($_POST['acg_post_status']) : 'publish';
    $allowed = ['publish','draft','pending'];
    if (!in_array($status, $allowed, true)) $status = 'publish';
    update_post_meta($post_id, self::MK_POST_STATUS, $status);

    // Schedule
    $sv = isset($_POST['acg_schedule_value']) ? max(1, (int)$_POST['acg_schedule_value']) : 24;
    update_post_meta($post_id, self::MK_SCHEDULE_VALUE, $sv);

    $su = isset($_POST['acg_schedule_unit']) ? sanitize_key($_POST['acg_schedule_unit']) : 'hours';
    $allowed_units = ['seconds','hours','days'];
    if (!in_array($su, $allowed_units, true)) $su = 'hours';
    update_post_meta($post_id, self::MK_SCHEDULE_UNIT, $su);
  }

  public function maybe_generate_on_save($post_id, $post) {
    if ($post->post_status !== 'publish') return;
    if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) return;

    // Generate post awal sebanyak "Jumlah" (sesuai field Jumlah)
    $amount = (int)get_post_meta($post_id, self::MK_AMOUNT, true);
    if ($amount < 1) $amount = 1;

    $this->generate_posts_for_campaign_limited($post_id, $amount);

    // Kalau masih ada keyword tersisa, schedule tick berikutnya (1 post per tick)
    if ($this->campaign_still_has_work($post_id)) {
      $this->schedule_next_tick($post_id);
    }
  }

  private function schedule_seconds_for_campaign($campaign_id) {
    $value = (int)get_post_meta($campaign_id, self::MK_SCHEDULE_VALUE, true);
    $unit  = (string)get_post_meta($campaign_id, self::MK_SCHEDULE_UNIT, true);

    if ($value < 1) $value = 24;
    if ($unit === '') $unit = 'hours';

    switch ($unit) {
      case 'seconds': return $value;
      case 'days':    return $value * 86400;
      case 'hours':
      default:        return $value * 3600;
    }
  }

  private function schedule_next_tick($campaign_id) {
    $seconds = $this->schedule_seconds_for_campaign($campaign_id);
    $ts = time() + $seconds;

    // Hindari double schedule dengan args yang sama
    if (!wp_next_scheduled('acg_run_campaign_tick', [$campaign_id])) {
      wp_schedule_single_event($ts, 'acg_run_campaign_tick', [$campaign_id]);
    }
  }

  public function cron_run_campaign_tick($campaign_id) {
    $campaign = get_post($campaign_id);
    if (!$campaign || $campaign->post_type !== self::CPT) return;

    // Hanya jalan kalau masih publish
    if ($campaign->post_status !== 'publish') return;

    // Generate 1 post setiap tick
    $created = $this->generate_posts_for_campaign_limited($campaign_id, 1);

    // Kalau masih bisa generate lagi, schedule tick berikutnya
    if ($created > 0 && $this->campaign_still_has_work($campaign_id)) {
      $this->schedule_next_tick($campaign_id);
    }
  }

  private function campaign_still_has_work($campaign_id) {
    $keywords_raw = (string)get_post_meta($campaign_id, self::MK_KEYWORDS, true);
    $lines = array_values(array_filter(array_map('trim', explode("\n", str_replace(["\r\n","\r"], "\n", $keywords_raw)))));

    $used_titles = get_post_meta($campaign_id, self::MK_USED_TITLES, true);
    if (!is_array($used_titles)) $used_titles = [];

    foreach ($lines as $t) {
      if ($t !== '' && !in_array($t, $used_titles, true)) {
        return true;
      }
    }
    return false;
  }

  public function cleanup_campaign_cron_on_delete($post_id) {
    $p = get_post($post_id);
    if (!$p || $p->post_type !== self::CPT) return;

    $ts = wp_next_scheduled('acg_run_campaign_tick', [$post_id]);
    while ($ts) {
      wp_unschedule_event($ts, 'acg_run_campaign_tick', [$post_id]);
      $ts = wp_next_scheduled('acg_run_campaign_tick', [$post_id]);
    }
  }

  private function generate_posts_for_campaign_limited($campaign_id, $limit) {
    $keywords_raw = (string)get_post_meta($campaign_id, self::MK_KEYWORDS, true);
    $post_type    = (string)get_post_meta($campaign_id, self::MK_POST_TYPE, true);
    $cat_id       = (int)get_post_meta($campaign_id, self::MK_CATEGORY, true);
    $post_status  = (string)get_post_meta($campaign_id, self::MK_POST_STATUS, true);

    if ($post_type === '') $post_type = 'post';
    if ($post_status === '') $post_status = 'publish';

    $template = (string)get_option(self::OPT_TEMPLATE, '');
    if (trim($template) === '') return 0;

    $lines = array_values(array_filter(array_map('trim', explode("\n", str_replace(["\r\n","\r"], "\n", $keywords_raw)))));
    if (empty($lines)) return 0;

    $used_titles = get_post_meta($campaign_id, self::MK_USED_TITLES, true);
    if (!is_array($used_titles)) $used_titles = [];

    $generated_ids = get_post_meta($campaign_id, self::MK_GEN_IDS, true);
    if (!is_array($generated_ids)) $generated_ids = [];

    $created = 0;

    foreach ($lines as $title) {
      if ($created >= $limit) break;
      if ($title === '') continue;
      if (in_array($title, $used_titles, true)) continue;

      // Cegah duplikat global untuk post_type yang sama
      $existing = get_page_by_title($title, OBJECT, $post_type);
      if ($existing) {
        $used_titles[] = $title;
        continue;
      }

      $content = str_replace('{title}', $title, $template);
      $content = $this->parse_spintax($content);

      $new_id = wp_insert_post([
        'post_title'   => $title,
        'post_content' => $content,
        'post_status'  => $post_status,
        'post_type'    => $post_type,
      ], true);

      if (is_wp_error($new_id)) continue;

      // Set category jika didukung
      if ($cat_id > 0 && taxonomy_exists('category')) {
        $taxes = get_object_taxonomies($post_type, 'names');
        if (in_array('category', $taxes, true)) {
          wp_set_post_terms($new_id, [$cat_id], 'category', false);
        }
      }

      $used_titles[] = $title;
      $generated_ids[] = (int)$new_id;
      $created++;
    }

    update_post_meta($campaign_id, self::MK_USED_TITLES, $used_titles);
    update_post_meta($campaign_id, self::MK_GEN_IDS, $generated_ids);

    return $created;
  }

  /**
   * Spintax parser sederhana: {a|b|c}
   * Loop sampai stabil (dibatasi agar aman).
   */
  private function parse_spintax($text) {
    $max_loops = 60;
    $pattern = '/\{([^{}]+)\}/';

    for ($i = 0; $i < $max_loops; $i++) {
      if (!preg_match($pattern, $text)) break;

      $text = preg_replace_callback($pattern, function($m) {
        $choices = explode('|', $m[1]);
        return $choices[array_rand($choices)];
      }, $text);
    }
    return $text;
  }
}

new ACG_Plugin();
